/*! \file tlk_cert_api_crys.h 
**********************************************************************************
*Title:                         Discretix Certificate Handling Toolkit source file
*
* Filename:                     tlk_cert_api_crys.h 
*
* Project, Target, subsystem:   Toolkit, Certificate Handling
* 
* Created:                      29.01.2008
*
* Modified:                     29.01.2008
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2008 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_CERT_API_CRYS_H_
#define _TLK_CERT_API_CRYS_H_
#include "tlk_cert_types.h"
/**
* @brief    This API will verify an RSA signature on a block of data using the public key from a CVPO, 
*           after verifying that the keyUsage kept in the proof includes the digitalSignature bit. 
*
* @param userContext_ptr        [in] - A pointer to the public Context, for the use of the function as a space to work on
* @param cvpoObject_ptr         [in] - A pointer to the public key data structure of the user.
* @param hashFunction           [in] - The hash function to be used. 
*                                      Currently only CRYS_PKCS1_SHA1 and CRYS_PKCS1_MD5
*                                      are supported. The other hash functions recommended by PKCS#1 v2.1 are
*                                      SHA-256/284/512.  If PKCS1_ver is CRYS_PKCS1_VER15, then
*                                      the value CRYS_RSA_After_HASH_NOT_KNOWN_mode will allow
*                                      the signature data to determine the hash function to be used.
* @param maskGenFunction        [in] - The mask generation function. 
*                                      only for PKCS#1 v2.1 defines MGF1, so the only value allowed for v2.1 
*                                      is CRYS_PKCS1_MGF1. 
* @param saltLenfthInBytes      [in] - The Length of the Salt buffer. 
*                                      relevant for PKCS#1 Ver 2.1 Only Typical lengths are 0 and hLen (20 for SHA1)
*  						               The maximum length allowed is NSize - hLen - 2
* @param dataIn_ptr             [in] - A pointer to the data whose signature is
*                                      to be verified.
* @param dataInSizeInBytes      [in] - The size, in bytes, of the data whose signature is to be verified.
* @param signature_ptr          [in] - A pointer to the signature to be verified. 
*                                      The length of the signature is PubKey_ptr->N.len bytes 
*                                     (that is, the size of the modulus, in bytes).
* @param pkcs1Version           [in] - Ver 1.5 or 2.1, according to the functionality required
*
* @return TLK_CERT_RC_OK	                 - The operation completed successfully.
* @return TLK_CERT_RC_FAIL                   - The operation failed.
*/
DxError_t TLK_CERT_CRYS_RSASignatureVerify(                 const   CRYS_RSAPubUserContext_t    *userContext_ptr,
                                                            const   TLK_CERT_CPVO_t             *cvpoObject_ptr,
                                                            const   CRYS_RSA_HASH_OpMode_t       hashFunction,
                                                            const   CRYS_PKCS1_MGF_t             maskGenFunction,
                                                            const   DxUint16_t                   saltLenfthInBytes,
                                                            const   DxUint8_t                   *dataIn_ptr,
                                                                    DxUint32_t                   dataInSizeInBytes,
                                                            const   DxUint8_t                   *signature_ptr,
                                                            const   CRYS_PKCS1_version           pkcs1Version);
/**
* @brief    This API will encrypt (RSA) a block of data using the public key from a CVPO, 
*           after verifying that the keyUsage kept in the proof includes the dataEncipherment bit.
*
* @param cvpoObject_ptr         [in]  - Pointer to the CVPO object.
*                                       We will extract pointer to the public key data structure of the User.
* @param primeData_ptr          [in]  - A pointer to a CRYS_RSAPrimeData_t that is used 
*                                       for the Encryption operation
* @param hashFunction           [in]  - The hash function to be used. 
*                                       Currently only CRYS_PKCS1_SHA1 is supported. 
*                                       The other hash functions recommended 
*                                       by PKCS#1 v2.1 are SHA-256/284/512.
* @param l_ptr                  [in]  - The label input. 
*                                       Version 2.1 of the standard defines L as the empty string, 
*                                       so this argument is currently ignored. 
* @param lLengthInBytes         [in]  - The label length. 
*                                       Version 2.1 of the standard defines L as the empty string, 
*                                       so this argument is currently ignored.
* @param maskGenFunction        [in]  -  The mask generation function. 
*                                        PKCS#1 v2.1 defines MGF1, so the only value allowed here is CRYS_PKCS1_MGF1.
* @param dataIn_ptr             [in]  - A pointer to the data to encrypt.
* @param dataInSize             [in]  - The size, in bytes, of the data to encrypt. 
*                                       @note Must be <= (modulus_size - 2*Hash_output_length - 2).
* @param output_ptr             [out] - A pointer to the encrypted data. 
*                                       The buffer must be at least PubKey_ptr->N.len bytes long 
*                                       (that is, the size of the modulus, in bytes).
* @param pkcs1Version           [in]  - Ver 1.5 or 2.1, according to the functionality required
*
* @return TLK_CERT_RC_OK	          - The operation completed successfully.
* @return TLK_CERT_RC_FAIL            - The operation failed.
*/
DxError_t TLK_CERT_CRYS_RSABlockEncrypt(                    const   TLK_CERT_CPVO_t             *cvpoObject_ptr,
                                                            const   CRYS_RSAPrimeData_t         *primeData_ptr,
                                                            const   CRYS_RSA_HASH_OpMode_t       hashFunction,
                                                            const   DxUint8_t                   *l_ptr,
                                                            const   DxUint16_t                   lLengthInBytes,
                                                            const   CRYS_PKCS1_MGF_t             maskGenFunction,
                                                            const   DxUint8_t                   *dataIn_ptr,
                                                            const   DxUint16_t                   dataInSize,
                                                                    DxUint8_t                   *output_ptr,
                                                            const   CRYS_PKCS1_version           pkcs1Version);
#endif